package gov.va.med.mhv.getcare.validator;

import java.util.Date;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.getcare.common.dto.HealthInsuranceDTO;
import gov.va.med.mhv.getcare.model.HealthInsurance;
import gov.va.med.mhv.getcare.repository.HealthInsuranceRepository;
import gov.va.med.mhv.getcare.util.MessagesUtil;

@Component
public class HealthInsuranceValidator extends Validator{
	private static final int INSURANCEIDNUMBER_MAX_LENGTH = 30;
	private static final int COMMENTS_MAX_LENGTH = 255;
	private static final int GROUPNUMBER_MAX_LENGTH = 30;
	private static final int COMPANYNAME_MIN_LENGTH = 2;
	private static final int COMPANYNAME_MAX_LENGTH = 50;
	private static final int FIRSTNAME_MIN_LENGTH = 2;
	private static final int FIRSTNAME_MAX_LENGTH = 25;
	private static final int LASTNAME_MIN_LENGTH = 2;
	private static final int LASTNAME_MAX_LENGTH = 25;
	
	@Autowired
	private HealthInsuranceRepository insuranceRepository;
	
	@Autowired
	private MessagesUtil messages;
	
	public void insuranceValidations(ResponseUtil response, HealthInsuranceDTO insuranceDTO) {
		fieldValidation(response, insuranceDTO);
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, insuranceDTO);
		}
	}

	private void complexValidation(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		validateSinglePrimaryInsurance(response,insuranceDTO);
		validateStartStopDate(response,insuranceDTO);
		
	}

	
	private void validateStartStopDate(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		if(insuranceDTO.getStartDate() !=null && insuranceDTO.getStopDate() != null){
			if(insuranceDTO.getStopDate().before(insuranceDTO.getStartDate())){
				response.getValidationErrors().put("invalid stop date",messages.getInvalidStopDate());
			}
		}
		
	}

	private void validateSinglePrimaryInsurance(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		if(insuranceDTO.getPrimaryInd()){
			List<HealthInsurance> insurances=insuranceRepository.findPrimaryInsurarance(insuranceDTO.getUserProfileId());
			if(insurances != null && insurances.size() > 0){
				for(HealthInsurance insurance: insurances){
					if(insuranceDTO.getHealthInsuranceId() != null){
					   if(insuranceDTO.getHealthInsuranceId().longValue() != insurance.getHealthInsuranceId().longValue()){
						   response.getValidationErrors().put("invalid Primary Insurance",messages.getInvalidPrimaryInsurance());
						   break;	
					   }
					}else{
						   response.getValidationErrors().put("invalid Primary Insurance",messages.getInvalidPrimaryInsurance());
						break;
					}
				}
			}
		}
		
	}

	private void fieldValidation(ResponseUtil response,	HealthInsuranceDTO insuranceDTO) {
		validateInsuranceIdNumber(response,insuranceDTO);
		validateComments(response,insuranceDTO);
		validateStartDate(response,insuranceDTO);
		validatePreApprovalPhoneNumber(response,insuranceDTO);
		validateGroupNumber(response,insuranceDTO);
		validateCompanyPhoneNumber(response,insuranceDTO);
		validateStopDate(response,insuranceDTO);
		validateCompanyName(response,insuranceDTO);
		validateFirstName(response,insuranceDTO);
		validatePrimaryInd(response,insuranceDTO);
		validateLastName(response,insuranceDTO);
		validateUserProfile(response,insuranceDTO);
	}

	private void validateUserProfile(ResponseUtil response,	HealthInsuranceDTO insuranceDTO) {
		if( insuranceDTO.getUserProfileId() == null ) {
			response.getValidationErrors().put("userProfile", "userProfile is required");
			return;
		}
		boolean valid= isUserProfileValid(insuranceDTO.getUserProfileId() );
		if ( !valid) {
			response.getValidationErrors().put("userProfile", "userProfile is Invalid");
		}
		
		
	}

	private void validateLastName(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getLastNameOfInsured();
		if ( value == null ) {
			Validations.validateRequired("Last Name of Insured", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMinLength("Last Name of Insured",value, LASTNAME_MIN_LENGTH, response.getValidationErrors());
		Validations.validateMaxLength("Last Name of Insured",value, LASTNAME_MAX_LENGTH, response.getValidationErrors());
		
	}

	private void validatePrimaryInd(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		Boolean value = insuranceDTO.getPrimaryInd();
		if ( value == null ) {
			Validations.validateRequired("Primary Insurance Provider", value, response.getValidationErrors());
			return;
		}	
	}

	private void validateFirstName(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getFirstNameOfInsured();
		if ( value == null ) {
			Validations.validateRequired("First Name of Insured", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMinLength("First Name of Insured",value, FIRSTNAME_MIN_LENGTH, response.getValidationErrors());
		Validations.validateMaxLength("First Name of Insured",value, FIRSTNAME_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateCompanyName(ResponseUtil response,	HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getCompanyName();
		if ( value == null ) {
			Validations.validateRequired("Health Insurance Company", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMinLength("Health Insurance Company",value, COMPANYNAME_MIN_LENGTH, response.getValidationErrors());
		Validations.validateMaxLength("Health Insurance Company",value, COMPANYNAME_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateStopDate(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		Date value = insuranceDTO.getStopDate();
		if ( value == null ) {
			return;
		}	
	}

	private void validateCompanyPhoneNumber(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getCompanyPhone();
		if ( value == null ) {
			return;
		}	
		Validations.validatePhoneNumber("Health Insurance Company Phone Number", value, response.getValidationErrors());
		
	}

	private void validateGroupNumber(ResponseUtil response,	HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getGroupNumber();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Group Number",value, GROUPNUMBER_MAX_LENGTH, response.getValidationErrors());		
	}

	private void validatePreApprovalPhoneNumber(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getPreApprovalPhone();
		if ( value == null ) {
			return;
		}	
		Validations.validatePhoneNumber("Pre-Approval Phone Number", value, response.getValidationErrors());
	}

	private void validateStartDate(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		Date value = insuranceDTO.getStartDate();
		if ( value == null ) {
			Validations.validateRequired("Start Date", value, response.getValidationErrors());
			return;
		}	
	
	}

	private void validateComments(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
		
	}

	private void validateInsuranceIdNumber(ResponseUtil response,HealthInsuranceDTO insuranceDTO) {
		String value = insuranceDTO.getInsuranceIdNumber();
		if ( value == null ) {
			Validations.validateRequired("ID Number", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMaxLength("ID Number",value, INSURANCEIDNUMBER_MAX_LENGTH, response.getValidationErrors());
	}
}
